/*==========================================================================
* XFER.C - Copyright (c) 1993-94 ATI Technologies Inc. All rights reserved *
*                                                                          *
* Functions to transfer data between the CPU and video memory.             *
* ======================================================================== */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"
#include "..\inc\atim64.h"

void pgl_xferf(void far *cpuptr,
               unsigned int VGAoffset,
               unsigned int nbytes,
               int direction);

/* --------------------------------------------------------------------------
  PGL_writeblock - transfer data from a CPU buffer to video memory

  This function transfers 'nbytes' bytes of data from a CPU buffer (pointed
  to by 'cpuptr') to the video memory buffer starting at offset
  'memory_offset'. Both aperture methods are supported (VGA and Linear).
  However, the VGA aperture has priority if it is enabled. Note that the
  transfer is contigious.

  SRC    = cpuptr (segment:offset)
  DST    = memory_offset in bytes
  Length = nbytes (in bytes)

  Returns NO_ERROR for no error occurrences; otherwise YES_ERROR is returned.
-------------------------------------------------------------------------- */
int PGL_writeblock(void huge *cpuptr,
                   unsigned long memory_offset,
                   unsigned long nbytes)
{
    unsigned long save_page_reg;
    unsigned long page;
    unsigned long page_offset;
    unsigned long page_remainder;
    unsigned long bytes;
    unsigned long moffset;
    unsigned long remainder;
    unsigned char huge *ptr;
    unsigned char huge *hostptr;
    unsigned int word;
    unsigned char byte;
    int i;
    int fullxfers;

    // return error if 0 bytes to transfer
    if (nbytes == 0)
    {
        return (YES_ERROR);
    }

    hostptr = (unsigned char huge *)(cpuptr);
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // save VGA aperture write page register
        save_page_reg = regr(MEM_VGA_WP_SEL);

        // determine starting page and page offset (each page = 32768 bytes)
        page = memory_offset / 32768;
        page_offset = memory_offset - (page * 32768);

        // determine remainder within first 64K write page so that the rest
        // will be full size transfers
        page_remainder = 65536 - page_offset;

        // setup starting page (combine two 32K write pages together to form
        // one 64K write page)
        regw(MEM_VGA_WP_SEL, ((page + 1) << 16) | page);

        // do transfer if nbytes will fit in current page setup
        if (nbytes <= page_remainder)
        {
            bytes = nbytes;
            if (nbytes == 65536)
            {
                // does upto 65535 bytes
                pgl_xferf((void far *)hostptr,
                          (unsigned int)page_offset,
                          (unsigned int)65535,
                          1);

                // copy last byte
                byte = *((unsigned char huge *)(hostptr + 65535));
                *((unsigned char far *)(0xa0000000 + 65535)) = byte;
            }
            else
            {
                // copy upto 65535 bytes
                pgl_xferf((void far *)hostptr,
                          (unsigned int)page_offset,
                          (unsigned int)bytes,
                          1);
            }
        }
        else
        {
            // determine number of full size transfers (65536 bytes)
            bytes = nbytes - page_remainder;
            fullxfers = (int)(bytes / 65536);
            remainder = bytes - ((unsigned long)fullxfers * 65536);

            // transfer page remainder block first
            if (page_remainder == 65536)
            {
                // copy 65535 bytes
                pgl_xferf((void far *)(hostptr),
                          0,
                          (unsigned int)65535,
                          1);

                // copy last byte
                byte = *((unsigned char huge *)(hostptr + 65535));
                *((unsigned char far *)(0xa0000000 + page_offset + 65535)) = byte;
            }
            else
            {
                // copy upto 65535 bytes
                pgl_xferf((void far *)(hostptr),
                          0,
                          (unsigned int)page_remainder,
                          1);
            }

            moffset = page_remainder;

            // do full size transfers if not zero
            i = 0;
            while (i < fullxfers)
            {
                // advance to next 64K page (two 32K pages)
                page = page + 2;
                regw(MEM_VGA_WP_SEL, ((page + 1) << 16) | page);

                // copy 65535 bytes
                pgl_xferf((void far *)(hostptr + moffset),
                          0,
                          (unsigned int)65535,
                          1);

                // copy last byte
                byte = *((unsigned char huge *)(hostptr + moffset + 65535));
                *((unsigned char far *)(0xa0000000 + 65535)) = byte;

                // setup next 65536 byte transfer
                moffset = moffset + 65536;
                i++;
            }

            // transfer remainder
            if (remainder != 0)
            {
                page = page + 2;
                regw(MEM_VGA_WP_SEL, ((page + 1) << 16) | page);

                // copy upto 65535 bytes
                pgl_xferf((void far *)(hostptr + moffset),
                          0,
                          (unsigned int)remainder,
                          1);
            }
        }

        // restore VGA aperture write page register
        regw(MEM_VGA_WP_SEL, save_page_reg);
    }
    else
    {
        // determine number of full size transfers (65536 bytes)
        fullxfers = (int)(nbytes / 65536);
        remainder = nbytes - ((unsigned long)fullxfers * 65536);

        // do full size transfers
        ptr = hostptr;
        moffset = memory_offset;
        i = 0;
        while (i < fullxfers)
        {
            if (pgl_movemem((void far *)(ptr),
                            PGL_modecfg.aperture_address + moffset,
                            (unsigned int)32768, MEM_WRITE) != NO_ERROR)
            {
                return (YES_ERROR);
            }
            moffset = (unsigned long)(moffset + 65536);
            ptr = (unsigned char huge *)(ptr + 65536);
            i++;
        }

        // do remainder transfer (unless size is 1)
        if (remainder != 1)
        {
            if (pgl_movemem((void far *)(ptr),
                            PGL_modecfg.aperture_address + moffset,
                            (unsigned int)(remainder / 2), MEM_WRITE) != NO_ERROR)
            {
                return (YES_ERROR);
            }
            moffset = (unsigned long)(moffset + (remainder & 0xfffffffe));
            ptr = (unsigned char huge *)(ptr + (remainder & 0xfffffffe));
        }

        // handle odd last byte case or if size is 1 byte
        if (((remainder & 1) == 1) || (remainder == 1))
        {
            // do read-modify-write since only words can be transferred
            if (pgl_movemem((void far *)(&word),
                            PGL_modecfg.aperture_address + moffset,
                            1, MEM_READ) != NO_ERROR)
            {
                return (YES_ERROR);
            }
            byte = *((unsigned char huge *)(ptr));
            word = (unsigned int)((word & 0xff00) | (byte & 0xff));
            if (pgl_movemem((void far *)(&word),
                            PGL_modecfg.aperture_address + moffset,
                            1, MEM_WRITE) != NO_ERROR)
            {
                return (YES_ERROR);
            }
        }
    }

    return (NO_ERROR);
}

/* --------------------------------------------------------------------------
  PGL_readblock - transfer data from the video memory to a CPU buffer

  This function transfers 'nbytes' bytes of data from the video memory buffer
  starting at offset 'memory_offset' to a CPU buffer (pointed to by
  'cpuptr'). Both aperture methods are supported (VGA and Linear). However,
  the VGA aperture has priority if it is enabled. Note that the transfer is
  contigious.

  SRC    = memory_offset in bytes
  DST    = cpuptr (segment:offset)
  Length = nbytes (in bytes)

  Returns NO_ERROR for no error occurrences; otherwise YES_ERROR is returned.
-------------------------------------------------------------------------- */
int PGL_readblock(void huge *cpuptr,
                  unsigned long memory_offset,
                  unsigned long nbytes)
{
    unsigned long save_page_reg;
    unsigned long page;
    unsigned long page_offset;
    unsigned long page_remainder;
    unsigned long bytes;
    unsigned long moffset;
    unsigned long remainder;
    unsigned char huge *ptr;
    unsigned char huge *hostptr;
    unsigned int word;
    unsigned char byte;
    int i;
    int fullxfers;

    // return error if 0 bytes to transfer
    if (nbytes == 0)
    {
        return (YES_ERROR);
    }

    hostptr = (unsigned char huge *)(cpuptr);
    if (PGL_modecfg.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        // save VGA aperture write page register
        save_page_reg = regr(MEM_VGA_WP_SEL);

        // determine starting page and page offset (each page = 32768 bytes)
        page = memory_offset / 32768;
        page_offset = memory_offset - (page * 32768);

        // determine remainder within first 64K read page so that the rest
        // will be full size transfers
        page_remainder = 65536 - page_offset;

        // setup starting page (combine two 32K read pages together to form
        // one 64K read page)
        regw(MEM_VGA_RP_SEL, ((page + 1) << 16) | page);

        // do transfer if nbytes will fit in current page setup
        if (nbytes <= page_remainder)
        {
            bytes = nbytes;
            if (nbytes == 65536)
            {
                // does upto 65535 bytes
                pgl_xferf((void far *)hostptr,
                          (unsigned int)page_offset,
                          (unsigned int)65535,
                          0);

                // copy last byte
                byte = *((unsigned char far *)(0xa0000000 + 65535));
                *((unsigned char huge *)(hostptr + 65535)) = byte;
            }
            else
            {
                // copy upto 65535 bytes
                pgl_xferf((void far *)hostptr,
                          (unsigned int)page_offset,
                          (unsigned int)bytes,
                          0);
            }
        }
        else
        {
            // determine number of full size transfers (65536 bytes)
            bytes = nbytes - page_remainder;
            fullxfers = (int)(bytes / 65536);
            remainder = bytes - ((unsigned long)fullxfers * 65536);

            // transfer page remainder block first
            if (page_remainder == 65536)
            {
                // copy 65535 bytes
                pgl_xferf((void far *)(hostptr),
                          0,
                          (unsigned int)65535,
                          0);

                // copy last byte
                byte = *((unsigned char far *)(0xa0000000 + page_offset + 65535));
                *((unsigned char huge *)(hostptr + 65535)) = byte;
            }
            else
            {
                // copy upto 65535 bytes
                pgl_xferf((void far *)(hostptr),
                          0,
                          (unsigned int)page_remainder,
                          0);
            }

            moffset = page_remainder;

            // do full size transfers if not zero
            i = 0;
            while (i < fullxfers)
            {
                // advance to next 64K page (two 32K pages)
                page = page + 2;
                regw(MEM_VGA_RP_SEL, ((page + 1) << 16) | page);

                // copy 65535 bytes
                pgl_xferf((void far *)(hostptr + moffset),
                          0,
                          (unsigned int)65535,
                          0);

                // copy last byte
                byte = *((unsigned char far *)(0xa0000000 + 65535));
                *((unsigned char huge *)(hostptr + moffset + 65535)) = byte;

                // setup next 65536 byte transfer
                moffset = moffset + 65536;
                i++;
            }

            // transfer remainder
            if (remainder != 0)
            {
                page = page + 2;
                regw(MEM_VGA_RP_SEL, ((page + 1) << 16) | page);

                // copy upto 65535 bytes
                pgl_xferf((void far *)(hostptr + moffset),
                          0,
                          (unsigned int)remainder,
                          0);
            }
        }

        // restore VGA aperture write page register
        regw(MEM_VGA_WP_SEL, save_page_reg);
    }
    else
    {
        // determine number of full size transfers (65536 bytes)
        fullxfers = (int)(nbytes / 65536);
        remainder = nbytes - ((unsigned long)fullxfers * 65536);

        // do full size transfers
        ptr = hostptr;
        moffset = memory_offset;
        i = 0;
        while (i < fullxfers)
        {
            if (pgl_movemem((void far *)(ptr),
                            PGL_modecfg.aperture_address + moffset,
                            (unsigned int)32768, MEM_READ) != NO_ERROR)
            {
                return (YES_ERROR);
            }
            moffset = (unsigned long)(moffset + 65536);
            ptr = (unsigned char huge *)(ptr + 65536);
            i++;
        }

        // do remainder transfer (unless size is 1)
        if (remainder != 1)
        {
            if (pgl_movemem((void far *)(ptr),
                            PGL_modecfg.aperture_address + moffset,
                            (unsigned int)(remainder / 2), MEM_READ) != NO_ERROR)
            {
                return (YES_ERROR);
            }
            moffset = (unsigned long)(moffset + (remainder & 0xfffffffe));
            ptr = (unsigned char huge *)(ptr + (remainder & 0xfffffffe));
        }

        // handle odd last byte case or if size is 1 byte
        if (((remainder & 1) == 1) || (remainder == 1))
        {
            // mask off upper byte since only words can be transferred
            if (pgl_movemem((void far *)(&word),
                            PGL_modecfg.aperture_address + moffset,
                            1, MEM_READ) != NO_ERROR)
            {
                return (YES_ERROR);
            }
            *((unsigned char huge *)(ptr)) = (unsigned char)(word & 0xff);
        }
    }

    return (NO_ERROR);
}

