/*==========================================================================
* BLIT.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved    *
*                                                                          *
* PGL functions to perform BLITs.                                          *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_getbuffersize - calculate the number of dwords to allocate when
                      performing a screen to buffer blit or vice versa

  This function is used in conjunction with the screen to buffer and buffer
  to screen blit functions. The given size of the blit rectangle is used
  to calculate the number of dwords to allocate in CPU memory. The returned
  value will be slightly conservative to prevent overwriting of the allocated
  CPU memory area.
-------------------------------------------------------------------------- */
unsigned long PGL_getbuffersize(int recwidth, int recheight)
{
    unsigned long totaldwords, totalbits;

    totalbits = (unsigned long)recheight;
    totalbits = (unsigned long)(totalbits * recwidth);
    totalbits = (unsigned long)(totalbits * PGL_modecfg.bpp);
    totaldwords = totalbits / 32;
    if (totalbits > (totaldwords * 32))
    {
        totaldwords++;
    }

    return (totaldwords);
}

/* --------------------------------------------------------------------------
  PGL_screentoscreenblit - perform a blit from the source to destination
                           rectangle

  A color screen to screen blit is performed by copying the contents of the
  source rectangle to the destination rectangle. The current mixes are used.
  This function also handles overlapping blit regions.
-------------------------------------------------------------------------- */
void PGL_screentoscreenblit(int srcleft, int srctop, int dstleft, int dsttop,
                            int recwidth, int recheight)
{
    unsigned long temp1, temp2, temp3;
    unsigned long rotation;
    unsigned long direction;
    int srcx, srcy, dstx, dsty;

    // determine if blit regions overlap - if not, skip to speed up
    // non-overlap case
    if ((abs(dstleft - srcleft) < recwidth) && (abs(dsttop - srctop) < recheight))
    {
        // determine blit direction
        direction = 0;
        if (srcleft < dstleft)
        {
            direction = direction | DST_X_RIGHT_TO_LEFT;
            if (PGL_modecfg.bpp == 24)
            {
                // special case for 24 bpp modes
                srcx = srcleft + recwidth;
                dstx = dstleft + recwidth;
            }
            else
            {
                srcx = srcleft + recwidth - 1;
                dstx = dstleft + recwidth - 1;
            }
        }
        else
        {
            direction = direction | DST_X_LEFT_TO_RIGHT;
            srcx = srcleft;
            dstx = dstleft;
        }
        if (srctop < dsttop)
        {
            direction = direction | DST_Y_BOTTOM_TO_TOP;
            srcy = srctop + recheight - 1;
            dsty = dsttop + recheight - 1;
        }
        else
        {
            direction = direction | DST_Y_TOP_TO_BOTTOM;
            srcy = srctop;
            dsty = dsttop;
        }
    }
    else
    {
        // standard blit for non-overlapping regions
        direction = DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT;
        srcx = srcleft;
        srcy = srctop;
        dstx = dstleft;
        dsty = dsttop;
    }

    // adjust X coordinates and widths if 24 bpp mode
    rotation = 0;
    if (PGL_modecfg.bpp == 24)
    {
        rotation = pgl_get24bpprotation(dstleft);
        srcx = srcx * 3;
        dstx = dstx * 3;
        recwidth = recwidth * 3;
    }

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(DP_SRC);
    temp2 = regr(SRC_CNTL);
    temp3 = regr(DST_CNTL);

    // implement screen to screen blit
    regw(DP_SRC, FRGD_SRC_BLIT);
    regw(SRC_CNTL, 0);
    regw(DST_CNTL, (temp3 & 0x80) | rotation | direction |
                   DST_Y_TILE | DST_X_TILE);
    regw(SRC_Y_X, ((unsigned long)(srcx) << 16) | srcy);
    regw(SRC_HEIGHT1_WIDTH1, ((unsigned long)(recwidth) << 16) | recheight);
    regw(DST_Y_X, ((unsigned long)(dstx) << 16) | dsty);
    regw(DST_HEIGHT_WIDTH, ((unsigned long)(recwidth) << 16) | recheight);

    // restore
    PGL_waitforfifo(3);
    regw(DP_SRC, temp1);
    regw(SRC_CNTL, temp2);
    regw(DST_CNTL, temp3);
}

/* --------------------------------------------------------------------------
  PGL_ncscreentoscreenblit - perform a non-conforming blit from the source
                             to destination

  A color screen to screen non-conforming blit is performed by copying the
  contents of the source to the destination. The current mixes are used. The
  main objective of this function is to save space when saving an on-screen
  rectanglar area and to expand the saved data when restoring.

  Input parameters:

    RECLEFT,    - describes the rectangle area (usually in on-screen memory)
    RECTOP,
    RECWIDTH,
    RECHEIGHT

    X, Y        - coordinate where rectangle area will be linearized
                  (usually in off-screen memory)

    SAVEFLAG    - SAVE_BLIT   : rectangle is source
                  RESTORE_BLIT: rectangle is destination

  Output parameters:

    END_POINT   - coordinate where linearized rectangle data ends

-------------------------------------------------------------------------- */
PGL_point PGL_ncscreentoscreenblit(int recleft, int rectop,
                                   int recwidth, int recheight,
                                   int x, int y,
                                   int saveflag)
{
    unsigned long temp1, temp2, temp3, temp4, temp5;
    unsigned long rotation, offset, pitch;
    unsigned long remainder, endoffset, lx, ly, linepitch;
    int qpixels;
    PGL_point endpoint;

    /* determine endpoints of rectanges */
    rotation = 0;
    if (PGL_modecfg.bpp == 24)
    {
        if (saveflag == RESTORE_BLIT)
        {
            rotation = pgl_get24bpprotation(recleft);
        }
        else
        {
            rotation = 0;
        }
        recleft = recleft * 3;
        recwidth = recwidth * 3;
    }

    // get closest pitch to recwidth
    qpixels = recwidth / 8;
    if (recwidth > (qpixels * 8))
    {
        qpixels++;
    }

    // 4 bpp mode pitch must be a multiple of 16 pixels
    if (PGL_modecfg.bpp == 4)
    {
        if ((qpixels & 1) == 1)
        {
            qpixels++;
        }
    }
    pitch = (unsigned long)qpixels;
    if (PGL_modecfg.bpp == 4)
    {
        remainder = (unsigned long)(pitch * recheight * 4);
    }
    else
    {
        remainder = (unsigned long)(pitch * recheight * PGL_modecfg.bpp);
    }
    if (PGL_modecfg.bpp == 24)
    {
        remainder = remainder / 3;
    }
    pitch = pitch << 22;

    // get dword offset pointing to linearized data area
    endoffset = pgl_getxyoffset(x, y);
    offset = endoffset / 8;

    // calculate end point coordinate of operation
    linepitch = (unsigned long)(PGL_modecfg.pitch);
    if (PGL_modecfg.bpp == 4)
    {
        linepitch = linepitch / 2;
    }
    else
    {
        linepitch = (unsigned long)(linepitch * (PGL_modecfg.bpp / 8));
    }
    endoffset = endoffset + remainder;
    ly = endoffset / linepitch;
    remainder = endoffset - (ly * linepitch);
    if (PGL_modecfg.bpp == 4)
    {
        lx = remainder * 2;
    }
    else
    {
        lx = (unsigned long)(remainder * (PGL_modecfg.bpp / 8));
    }
    endpoint.x = (int)(lx + 2);
    endpoint.y = (int)ly;
    if (endpoint.x >= PGL_modecfg.pitch)
    {
        endpoint.x = 0;
        (endpoint.y)++;
    }

    // if endpoint is greater than available memory, don't perform blit
    if (endpoint.y > PGL_modecfg.maxy)
    {
        return (endpoint);
    }

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(SRC_OFF_PITCH);
    temp2 = regr(DST_OFF_PITCH);
    temp3 = regr(DP_SRC);
    temp4 = regr(SRC_CNTL);
    temp5 = regr(DST_CNTL);

    // implement non-conforming screen to screen blit
    if (saveflag == RESTORE_BLIT)
    {
        regw(SRC_OFF_PITCH, pitch | offset);
        regw(DP_SRC, FRGD_SRC_BLIT);
        regw(SRC_CNTL, 0);
        regw(DST_CNTL, (temp5 & 0x80) | rotation |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
        regw(SRC_Y_X, 0);
        regw(SRC_HEIGHT1_WIDTH1, ((unsigned long)(recwidth) << 16) | recheight);
        regw(DST_Y_X, ((unsigned long)(recleft) << 16) | rectop);
        regw(DST_HEIGHT_WIDTH, ((unsigned long)(recwidth) << 16) | recheight);
    }
    else
    {
        regw(DST_OFF_PITCH, pitch | offset);
        regw(DP_SRC, FRGD_SRC_BLIT);
        regw(SRC_CNTL, 0);
        regw(DST_CNTL, (temp5 & 0x80) | rotation |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
        regw(SRC_Y_X, ((unsigned long)(recleft) << 16) | rectop);
        regw(SRC_HEIGHT1_WIDTH1, ((unsigned long)(recwidth) << 16) | recheight);
        regw(DST_Y_X, 0);
        regw(DST_HEIGHT_WIDTH, ((unsigned long)(recwidth) << 16) | recheight);
    }

    // restore
    PGL_waitforfifo(5);
    regw(SRC_OFF_PITCH, temp1);
    regw(DST_OFF_PITCH, temp2);
    regw(DP_SRC, temp3);
    regw(SRC_CNTL, temp4);
    regw(DST_CNTL, temp5);

    return (endpoint);
}

/* --------------------------------------------------------------------------
  PGL_buffertoscreenblit - perform a blit from a buffer to a destination
                           rectangle

  A color buffer to screen blit is performed by copying the contents of the
  source buffer to the destination rectangle. The host buffer data will be
  retrieved in a contiguous manner. The current mixes are used. The buffer
  is expected to contain packed LSB to MSB order data. For example, each
  double word contains 4 pixels in 8 bpp mode where bit 0 of the double word
  represents to first pixel drawn in a left to right direction.
-------------------------------------------------------------------------- */
void PGL_buffertoscreenblit(unsigned long huge *bufferptr,
                            int dstleft, int dsttop,
                            int recwidth, int recheight)
{
    int dstright, dstbottom;
    unsigned long dwordcount, totaldwords;
    unsigned long bufdata;
    unsigned long temp1, temp2;
    unsigned long rotation;
    int chkwrts;

    /* determine endpoints of rectanges */
    rotation = 0;
    if (PGL_modecfg.bpp == 24)
    {
        rotation = pgl_get24bpprotation(dstleft);
        dstleft = dstleft * 3;
        recwidth = recwidth * 3;
    }
    dstright = dstleft + recwidth;
    dstbottom = dsttop + recheight;

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(DP_SRC);
    temp2 = regr(DST_CNTL);

    // implement buffer to screen blit
    regw(DP_SRC, FRGD_SRC_HOST);
    regw(DST_CNTL, (temp2 & 0x80) | rotation |
                   DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT |
                   DST_Y_TILE | DST_X_TILE);
    regw(DST_Y_X, ((unsigned long)(dstleft) << 16) | dsttop);
    regw(DST_HEIGHT_WIDTH, ((unsigned long)(recwidth) << 16) | recheight);

    totaldwords = PGL_getbuffersize(recwidth, recheight);

    chkwrts = 8;
    for (dwordcount = 0; dwordcount < totaldwords; dwordcount++)
    {
        bufdata = *(bufferptr + dwordcount);
        if (PGL_modecfg.bpp == 4)
        {
            bufdata = pgl_swapnibble(bufdata);
        }
        regw(HOST_DATA0, bufdata);
        chkwrts++;
        if (chkwrts > 15)
        {
            PGL_waitforemptyfifo();
            chkwrts = 0;
        }
    }

    // restore
    PGL_waitforidle();
    regw(DP_SRC, temp1);
    regw(DST_CNTL, temp2);
}

/* --------------------------------------------------------------------------
  PGL_screentobufferblit - perform a blit from a source rectangle to a
                           buffer

  A color screen to buffer blit is performed by copying the contents of the
  source rectangle to the destination buffer. The screen data will be written
  to the host buffer area in a contiguous manner. The current mixes are used.
  This routine will fill the given buffer with packed data identical to the
  format for the PGL_buffertoscreenblit() routine. Since the Mach64 does
  not support reading screen data, the data is retrieved through either the
  vga or linear aperture.
-------------------------------------------------------------------------- */
void PGL_screentobufferblit(unsigned long huge *bufferptr,
                            int srcleft, int srctop,
                            int recwidth, int recheight)
{
    int i, j, k, l;
    int hostwrts, shifter, shifts;
    unsigned long data, totaldwords, dwordcount;
    unsigned long pixel1, pixel2;

    totaldwords = PGL_getbuffersize(recwidth, recheight);

    if (PGL_modecfg.bpp == 24)
    {
        i = srcleft;
        j = srctop;
        l = 0;
        for (dwordcount = 0; dwordcount < totaldwords; dwordcount++)
        {
            pixel1 = PGL_getpixel(i, j);

            i++;
            if (i >= (recwidth + srcleft))
            {
                i = srcleft;
                j++;
            }

            pixel2 = PGL_getpixel(i, j);

            switch (l)
            {
                case 0:
                    data = (pixel2 << 24) | pixel1;
                    break;

                case 1:
                    data = (pixel2 << 16) | (pixel1 >> 8);
                    break;

                case 2:
                    data = (pixel2 << 8) | (pixel1 >> 16);
                    i++;
                    if (i >= (recwidth + srcleft))
                    {
                        i = srcleft;
                        j++;
                    }
                    break;
            }

            *(bufferptr + dwordcount) = data;

            l++;
            if (l > 2)
            {
                l = 0;
            }
        }
    }
    else
    {
        hostwrts = 32 / PGL_modecfg.bpp;
        shifts = PGL_modecfg.bpp;

        i = srcleft;
        j = srctop;
        for (dwordcount = 0; dwordcount < totaldwords; dwordcount++)
        {
            data = 0;
            shifter = 0;
            for (k = 0; k < hostwrts; k++)
            {
                data = data | (PGL_getpixel(i, j) << shifter);
                shifter = shifter + shifts;
                i++;
                if (i >= (recwidth + srcleft))
                {
                    i = srcleft;
                    j++;
                    if (j >= (recheight + srctop))
                    {
                        k = hostwrts;
                    }
                }
            }
            *(bufferptr + dwordcount) = data;
        }
    }
}

/* --------------------------------------------------------------------------
  PGL_monotoscreenblit - copy a monochrome bitmap from a buffer to screen
                         memory

  This function takes a monochrome bitmap contained in the given buffer
  and copies it to (DSTLEFT, DSTTOP) in linear format. RECWIDTH * RECHEIGHT
  describes the total number of bits to transfer from the buffer to screen
  memory. Function PGL_monoexpandblit() is normally used to color expand
  the monochrome data to on-screen memory.
-------------------------------------------------------------------------- */
void PGL_monotoscreenblit(unsigned long huge *bufferptr,
                          int dstleft, int dsttop,
                          int recwidth, int recheight)
{
    unsigned long dwordcount, totaldwords;
    unsigned long bufdata;
    unsigned long temp1, temp2, temp3, temp4, temp5;
    unsigned long offset;
    int chkwrts;

    // 24 bpp modes are not supported
    if (PGL_modecfg.bpp == 24)
    {
        return;
    }

    // calculate destination dword offset
    offset = pgl_getxyoffset(dstleft, dsttop) / 8;

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(DST_OFF_PITCH);
    temp2 = regr(DP_PIX_WIDTH);
    temp3 = regr(DP_SRC);
    temp4 = regr(DP_MIX);
    temp5 = regr(DST_CNTL);

    // implement monochrome buffer to screen blit
    regw(DST_OFF_PITCH, (regr(DST_OFF_PITCH) & 0xffc00000) | offset);
    if (PGL_modecfg.bpp == 4)
    {
        // order must match CRTC handling for 4 bpp
        regw(DP_PIX_WIDTH, BYTE_ORDER_MSB_TO_LSB | HOST_1BPP | DST_1BPP);
    }
    else
    {
        regw(DP_PIX_WIDTH, BYTE_ORDER_LSB_TO_MSB | HOST_1BPP | DST_1BPP);
    }
    regw(DP_SRC, MONO_SRC_HOST | FRGD_SRC_FRGD_CLR);
    regw(DP_MIX, FRGD_MIX_ONE | BKGD_MIX_ZERO);
    regw(DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    regw(DST_Y_X, 0);
    regw(DST_HEIGHT_WIDTH, ((unsigned long)(recwidth * recheight) << 16) | 1);

    // calculate number of 32 bit host writes required
    totaldwords = (unsigned long)recwidth;
    totaldwords = ((unsigned long)(totaldwords * recheight) + 31) / 32;

    chkwrts = 9;
    for (dwordcount = 0; dwordcount < totaldwords; dwordcount++)
    {
        bufdata = *(bufferptr + dwordcount);
        if (PGL_modecfg.bpp == 4)
        {
            bufdata = pgl_convertpattern(bufdata);
        }
        regw(HOST_DATA0, bufdata);
        chkwrts++;
        if (chkwrts > 15)
        {
            PGL_waitforemptyfifo();
            chkwrts = 0;
        }
    }

    // restore
    PGL_waitforidle();
    regw(DST_OFF_PITCH, temp1);
    regw(DP_PIX_WIDTH, temp2);
    regw(DP_SRC, temp3);
    regw(DP_MIX, temp4);
    regw(DST_CNTL, temp5);
}

/* --------------------------------------------------------------------------
  PGL_monoexpandblit - perform a monochrome to color expansion blit

  This function blits monochrome data in screen memory (SRCLEFT, SRCTOP)
  and color expands the data to another region in screen memory (DSTLEFT,
  DSTTOP). The monochrome data must be in linear format (this data is
  usually uploaded to screen memory by function PGL_monotoscreenblit().
  The color expansion operation uses the current engine mix and colors.
  A "1" in the monochrome data is expanded using the foreground color path
  and a "0" in the monochrome data is expanded using the background color
  path.
-------------------------------------------------------------------------- */
void PGL_monoexpandblit(int srcleft, int srctop,
                        int dstleft, int dsttop,
                        int recwidth, int recheight)
{
    unsigned long temp1, temp2, temp3, temp4, temp5;
    unsigned long offset;

    // 24 bpp modes are not supported
    if (PGL_modecfg.bpp == 24)
    {
        return;
    }

    // calculate destination qword offset
    offset = pgl_getxyoffset(srcleft, srctop) / 8;

    // save vital regs
    PGL_waitforidle();
    temp1 = regr(SRC_OFF_PITCH);
    temp2 = regr(DP_PIX_WIDTH);
    temp3 = regr(DP_SRC);
    temp4 = regr(SRC_CNTL);
    temp5 = regr(DST_CNTL);

    // perform monochrome expansion blit
    regw(SRC_OFF_PITCH, (regr(SRC_OFF_PITCH) & 0xffc00000) | offset);
    if (PGL_modecfg.bpp == 4)
    {
        // order must match CRTC handling for 4 bpp
        regw(DP_PIX_WIDTH, (regr(DP_PIX_WIDTH) & 0x000000ff) |
                           BYTE_ORDER_MSB_TO_LSB);
    }
    else
    {
        regw(DP_PIX_WIDTH, (regr(DP_PIX_WIDTH) & 0x000000ff) |
                           BYTE_ORDER_LSB_TO_MSB);
    }
    regw(DP_SRC, MONO_SRC_BLIT | FRGD_SRC_FRGD_CLR);

    regw(SRC_CNTL, SRC_LINEAR_ENABLE);
    regw(SRC_Y_X, 0);
    regw(SRC_HEIGHT1_WIDTH1, ((unsigned long)(recwidth * recheight) << 16) | 1);

    regw(DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT |
                   DST_Y_TILE | DST_X_TILE);
    regw(DST_Y_X, ((unsigned long)(dstleft) << 16) | dsttop);
    regw(DST_HEIGHT_WIDTH, ((unsigned long)(recwidth) << 16) | recheight);

    // restore
    PGL_waitforidle();
    regw(SRC_OFF_PITCH, temp1);
    regw(DP_PIX_WIDTH, temp2);
    regw(DP_SRC, temp3);
    regw(SRC_CNTL, temp4);
    regw(DST_CNTL, temp5);
}

