/*==========================================================================
  LINAPP.C

  Example code to enable linear memory aperture and to write & read a
  block of screen memory data using the linear aperture.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

unsigned int buffer[200];

int main (int argc, char *argv[])
{
    int i, j;
    unsigned long appaddr;

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // Ensure that the linear aperture is enabled. First, attempt to enable
    // the 4MB aperture. If that fails, attempt to enable the 8MB aperture.
    if (enable_linear_aperture (APERTURE_4M_ENABLE) == APERTURE_ENABLE_FAILURE)
    {
        if (enable_linear_aperture (APERTURE_8M_ENABLE) ==
            APERTURE_ENABLE_FAILURE)
        {
            printf ("Could not enable linear apertures.\n");
            return (1);
        }
    }

    // Process the command line arguments to override default resolution
    // and color depth settings.
    process_command_line (argc, argv);

    // set an accelerator mode
    if (open_mode (gmode_res, PITCH_XRES, gclr_depth) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // set engine context
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // Get linear memory aperture base address - this is determined in
    // update_aperture_status() which is called by open_mode().

    appaddr = modeinfo.aperture_address;
    if (appaddr == 0)
    {
        // disable accelerator mode and switch back to VGA text mode
        close_mode ();

        printf ("Error - linear memory aperture has a zero address. Try\n");
        printf ("        running the INSTALL program.\n");
        return (1);
    }

    /*
        Writing to and reading from the linear aperture can be done in
        several ways. Since the linear aperture is located in 'extended
        memory' space, a real mode application must use extended memory
        services to access memory through the linear aperture. There are
        several services available:

            . system BIOS int 15h, function 87h
            . Dos Protected Mode Interface (DPMI)
            . eXtended Memory Services (XMS)
            . Virtual Control Program Interface (VCPI)

        The last three mentioned require protected mode memory managers that
        support these services. The first one mentioned will typically be
        available since it is supported by the system BIOS. This method is
        also the slowest and is not practical for full scale applications.
        It should be noted that EMS (Expanded Memory Services) is not
        practical for accessing graphics frame buffers.

        If the application is in protected mode, the linear aperture can be
        accessed easily with virtually no overhead.
    */

    // fill buffer with data and write to screen memory through linear aperture
    for (i = 0; i < 200; i++)
    {
        buffer[i] = i;
    }
    movemem ((void far *) buffer, appaddr, 200, MEM_WRITE);

    // clear buffer and read screen memory back
    for (i = 0; i < 200; i++)
    {
        buffer[i] = 0;
    }
    movemem ((void far *) buffer, appaddr, 200, MEM_READ);


    // disable accelerator mode and switch back to VGA text mode
    close_mode ();


    // dump buffer to VGA screen
    printf ("\nLINEAR APERTURE read dump - values should be 0 to C7h\n");
    j = 0;
    for (i = 0; i < 200; i++)
    {
        printf ("%04X ", buffer[i]);

        // display 10 values per line
        j++;
        if (j > 9)
        {
            j = 0;
            printf ("\n");
        }
    }
    printf ("\n");

    return (0);
}

