/*==========================================================================
 VTGA.C

 Routines to bring in a TGA file to video memory for the sample code examples.

 Copyright (c) 1993-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>

#include "..\util\atim64.h"
#include "..\util\sample.h"
#include "vtga.h"

/*---------------------------------------------------------------------------
    GET_TARGA_HEADER - load targa file header information

    Inputs      : filename,
                  pointer to header structure

    Return codes: SUCCESS    - success
                  OPEN_ERROR - can't open file
---------------------------------------------------------------------------*/
int get_targa_header (char *filename, TARGA_HEADER *header)
{
    FILE *TargaFile;

    // open targa file
    TargaFile = fopen (filename, "rb");
    if (TargaFile == NULL)
    {
        return (OPEN_ERROR);
    }

    // check for support ability (type 1, 8 bpp, color map)
    fread (header, sizeof (TARGA_HEADER), 1, TargaFile);

    fclose (TargaFile);

    return (SUCCESS);
}

/*---------------------------------------------------------------------------
    SET_TARGA_PALETTE - load a targa image color table and set the palette

    Inputs      : filename of targa image file

    Return codes: SUCCESS          - success
                  OPEN_ERROR       - can't open file
                  IMAGE_TYPE_ERROR - targa file does not have color table

    Notes       : - a valid set accelerator mode is assumed
                  - the image size is intended to be fairly small
                  - accelerator mode is assumed to be 8 bpp
---------------------------------------------------------------------------*/
int set_targa_palette (char *filename)
{
    FILE *TargaFile;
    TARGA_HEADER header;
    PALETTE entry;
    int i;

    // open targa file
    TargaFile = fopen (filename, "rb");
    if (TargaFile == NULL)
    {
        return (OPEN_ERROR);
    }

    // check for support ability (type 1, 8 bpp, color map)
    fread (&header, sizeof (TARGA_HEADER), 1, TargaFile);

    // support check for type 1, 8 bpp, color map
    if ((header.cm_type != 1) ||
        (header.image_type != 1) ||
        (header.pixel_depth != 8) ||
        (header.image_coord == 1) ||
        (header.image_coord == 3))
    {
        // return error if image type is not supported (no color table)
        fclose (TargaFile);
        return (IMAGE_TYPE_ERROR);
    }

    // read color map and setup palette for 8 bpp
    for (i = header.cm_origin; i < header.cm_length; i++)
    {
        entry.blue = fgetc (TargaFile) >> 2;
        entry.green = fgetc (TargaFile) >> 2;
        entry.red = fgetc (TargaFile) >> 2;
        set_palette (i, entry);
    }

    fclose (TargaFile);

    // return to caller
    return (SUCCESS);
}

/*---------------------------------------------------------------------------
    LOAD_TARGA - load a targa image into video memory

    Inputs      : filename of targa image file,
                  (x, y) starting position (top-left corner),
                  paletteflag (set or don't set for 8 bpp mode) -
                    NOSET_PALETTE or SET_PALETTE

    Return codes: SUCCESS          - success
                  OPEN_ERROR       - can't open file
                  MEMORY_ERROR     - not enough memory for input buffer
                  IMAGE_TYPE_ERROR - targa file is not supported

    Notes       : - a valid set accelerator mode is assumed
                  - the image size is intended to be fairly small
                  - accelerator mode is assumed to be 8 bpp
---------------------------------------------------------------------------*/
int load_targa (char *filename, int x, int y)
{
    FILE *TargaFile;
    TARGA_HEADER header;
    unsigned char *inbuffer;
    unsigned long temp1, temp2, temp3, temp4, data;
    unsigned long totalbytes, hostwrts;
    int drawx, drawy, j;
    int shifter, remainder;
    int adder, error;
    unsigned char r, g, b;
    unsigned char value;

    // open targa file
    TargaFile = fopen (filename, "rb");
    if (TargaFile == NULL)
    {
        return (OPEN_ERROR);
    }

    // check for support ability (type 1, 8 bpp, color map)
    fread (&header, sizeof (TARGA_HEADER), 1, TargaFile);

    // support check for type 1, 8 bpp, color map
    error = 0;
    if ((header.cm_type != 1) ||
        (header.image_type != 1) ||
        (header.pixel_depth != 8) ||
        (header.image_coord == 1) ||
        (header.image_coord == 3))
    {
        error++;
    }

    // support check for type 2, 24 bpp, no color map
    if ((header.cm_type != 0) ||
        (header.image_type != 2) ||
        (header.pixel_depth != 24) ||
        (header.image_coord == 1) ||
        (header.image_coord == 3))
    {
        error++;
    }

    // return error if image type is not supported
    if (error > 1)
    {
        fclose (TargaFile);
        return (IMAGE_TYPE_ERROR);
    }

    // skip over color table if 8 bpp image type
    if (header.pixel_depth == 8)
    {
        fseek (TargaFile, (unsigned long) (header.cm_length * 3), SEEK_CUR);
    }

    // open input buffer (image width * image pixel depth in bytes)
    inbuffer = (unsigned char *) malloc ((header.width + 1) *
                                         (header.pixel_depth / 8));
    if (inbuffer == NULL)
    {
        fclose (TargaFile);
        return (MEMORY_ERROR);
    }

    // determine if an additional host write is needed at the end of the loop
    remainder = 0;
    totalbytes = (unsigned long) (header.width) *
                 (unsigned long) (header.height);
    hostwrts = totalbytes / 4;
    if ((hostwrts * 4) != totalbytes)
    {
        remainder++;
    }

    // setup engine for buffer to screen host data transfer
    wait_for_idle ();
    temp1 = regr (DP_SRC);
    temp2 = regr (DP_MIX);
    temp3 = regr (SRC_CNTL);
    temp4 = regr (DST_CNTL);

    regw (DP_SRC, FRGD_SRC_HOST);
    regw (DP_MIX, FRGD_MIX_S | BKGD_MIX_S);
    regw (SRC_CNTL, 0);

    if (header.image_coord == 0)
    {
        // bottom left
        regw (DST_CNTL, DST_Y_BOTTOM_TO_TOP | DST_X_LEFT_TO_RIGHT);
        regw (DST_X, x);
        regw (DST_Y, y + header.height - 1);
        regw (DST_HEIGHT, header.height);
        regw (DST_WIDTH, header.width);
    }
    else
    {
        // top left
        regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_RIGHT_TO_LEFT);
        regw (DST_X, x + header.width - 1);
        regw (DST_Y, y);
        regw (DST_HEIGHT, header.height);
        regw (DST_WIDTH, header.width);
    }

    // main draw loop
    if (header.pixel_depth == 24)
    {
        adder = 3;
    }
    else
    {
        adder = 1;
    }
    j = 0;
    shifter = 0;
    data = 0;
    for (drawy = 0; drawy < header.height; drawy++)
    {
        // get next image scanline
        fread ((unsigned char *) inbuffer,
               header.width * (header.pixel_depth / 8), 1, TargaFile);

        // get next line of pixels from input buffer
        for (drawx = 0; drawx < (header.width * (header.pixel_depth / 8));
             drawx = drawx + adder)
        {
            if (header.pixel_depth == 24)
            {
                // get 24 bpp data and convert to 8 bpp
                g = (unsigned char) (*(inbuffer + drawx) & 0xFF);
                b = (unsigned char) (*(inbuffer + drawx + 1) & 0xFF);
                r = (unsigned char) (*(inbuffer + drawx + 2) & 0xFF);
                value = (unsigned char) ((r & 0xE0) | ((g >> 3) & 0x1C) |
                                         ((b >> 6) & 0x03));
            }
            else
            {
                // get 8 bit data
                value = (unsigned char) (*(inbuffer + drawx) & 0xFF);
            }

            data = data | ((unsigned long) (value) << shifter);
            shifter = shifter + 8;

            // accumulate 32 bits at a time before writing
            j++;
            if (j > 3)
            {
                wait_for_fifo (1);
                regw (HOST_DATA0, data);
                j = 0;
                shifter = 0;
                data = 0;
            }
        }
    }

    // write remaining data if needed
    if (remainder != 0)
    {
        wait_for_fifo (1);
        regw (HOST_DATA0, data);
    }

    // restore main engine context
    wait_for_idle ();
    regw (DP_SRC, temp1);
    regw (DP_MIX, temp2);
    regw (SRC_CNTL, temp3);
    regw (DST_CNTL, temp4);

    // close buffers and targa file
    free (inbuffer);
    fclose (TargaFile);

    // return to caller
    return (SUCCESS);
}

