/*==========================================================================
* WAIT.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved    *
*                                                                          *
* Engine idle and fifo status functions.                                   *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

// dump register storage
unsigned long fifo_stat;
unsigned long gui_stat;
unsigned long bus_cntl;
unsigned long config_cntl;

/* --------------------------------------------------------------------------
  PGL_waitforemptyfifo - wait for all fifo entries to be empty.

  The FIFO is 16 entries by 32 bits wide. This is a faster version of
  PGL_waitforfifo(16). Returns NO_ERROR or YES_ERROR.
-------------------------------------------------------------------------- */
int PGL_waitforemptyfifo(void)
{
    if (PGL_modecfg.fifo_check_control == FIFO_CHECK_NORMAL)
    {
        while ((regr(FIFO_STAT) & 0xffff) != FIFO_EMPTY) ;
        return (NO_ERROR);
    }

    if (PGL_modecfg.fifo_check_control == FIFO_CHECK_TIMEOUT)
    {
        return (pgl_fifotimeout(WAITFOREMPTYFIFO, 0));
    }
}

/* --------------------------------------------------------------------------
  PGL_waitforfifo - wait for n empty entries in the engine fifo.

  The FIFO is 16 entries by 32 bits wide. The 'entries' value must be
  1 to 16. Returns NO_ERROR or YES_ERROR.
-------------------------------------------------------------------------- */
int PGL_waitforfifo(int entries)
{
    if (PGL_modecfg.fifo_check_control == FIFO_CHECK_NORMAL)
    {
        while ((regr(FIFO_STAT) & 0xffff) > ((unsigned int)(0x8000 >> entries))) ;
        return (NO_ERROR);
    }

    if (PGL_modecfg.fifo_check_control == FIFO_CHECK_TIMEOUT)
    {
        return (pgl_fifotimeout(WAITFORFIFO, entries));
    }
}

/* --------------------------------------------------------------------------
  PGL_waitforidle - wait until the engine is idle. Returns NO_ERROR or
                    YES_ERROR.
-------------------------------------------------------------------------- */
int PGL_waitforidle(void)
{
    if (PGL_modecfg.fifo_check_control == FIFO_CHECK_NORMAL)
    {
        PGL_waitforemptyfifo();
        while((regr(GUI_STAT) & ENGINE_BUSY) != ENGINE_IDLE) ;
        return (NO_ERROR);
    }

    if (PGL_modecfg.fifo_check_control == FIFO_CHECK_TIMEOUT)
    {
        if (pgl_fifotimeout(WAITFOREMPTYFIFO, 0) == NO_ERROR)
        {
            return (pgl_fifotimeout(WAITFORIDLE, 0));
        }
        else
        {
            return (YES_ERROR);
        }
    }
}

/* --------------------------------------------------------------------------
  pgl_fifotimeout - timeout version of fifo and idle functions

  Inputs : function: WAITFOREMPTYFIFO (0) = PGL_waitforemptyfifo()
                     WAITFORFIFO (1)      = PGL_waitforfifo()
                     WAITFORIDLE (2)      = PGL_waitforidle()

           entries : only used for WAITFORFIFO

  Outputs: NO_ERROR or YES_ERROR
-------------------------------------------------------------------------- */
int pgl_fifotimeout(int function, int entries)
{
    unsigned int starttick, endtick;
    int stopflag, retval;

    stopflag = 0;
    retval = NO_ERROR;
    starttick = *((unsigned int far *)(DOS_TICK_ADDRESS));
    endtick = starttick;
    while (stopflag == 0)
    {
        // check appropriate engine condition
        switch(function)
        {
            case WAITFOREMPTYFIFO:
                // check if FIFO is empty
                if ((regr(FIFO_STAT) & 0xffff) == FIFO_EMPTY)
                {
                    stopflag = 1;
                }
                break;

            case WAITFORFIFO:
                // check if FIFO contains the given empty entries
                if ((regr(FIFO_STAT) & 0xffff) <= ((unsigned int)(0x8000 >> entries)))
                {
                    stopflag = 1;
                }
                break;

            case WAITFORIDLE:
                // check if engine is idle
                if ((regr(GUI_STAT) & ENGINE_BUSY) == ENGINE_IDLE)
                {
                    stopflag = 1;
                }
                break;
        }

        // check timeout value if engine condition has not been met
        if (stopflag == 0)
        {
            endtick = *((unsigned int far *)(DOS_TICK_ADDRESS));
            if (endtick < starttick)
            {
                if (((0xffff - starttick) + endtick) > FIFO_TIMEOUT)
                {
                    retval = YES_ERROR;
                    stopflag = 1;
                }
            }
            else
            {
                if ((endtick - starttick) > FIFO_TIMEOUT)
                {
                    retval = YES_ERROR;
                    stopflag = 1;
                }
            }
        }
    }

    // terminate if a timeout error occured and this action is required
    if (retval == YES_ERROR)
    {
        if (PGL_modecfg.fifo_action_control == FIFO_ACTION_TERMINATE)
        {
            // read status registers
            fifo_stat = regr(FIFO_STAT);
            gui_stat = regr(GUI_STAT);
            bus_cntl = ior32(ioBUS_CNTL);
            config_cntl = ior32(ioCONFIG_CNTL);

            // terminate PGL and application
            pgl_terminate();
        }
    }

    return (retval);
}

/* --------------------------------------------------------------------------
  pgl_terminate - Terminate PGL and application due to a timeout error
-------------------------------------------------------------------------- */
void pgl_terminate(void)
{
    // disable accelerator mode if one was set
    if (PGL_modeSet == 1)
    {
        PGL_closemode();
    }

    // close all apertures
    PGL_close();

    // display status registers
    printf("M64PGL application terminated due to TIMEOUT error.\n");
    printf("FIFO_STAT  : %08lx\n", fifo_stat);
    printf("GUI_STAT   : %08lx\n", gui_stat);
    printf("BUS_CNTL   : %08lx\n", bus_cntl);
    printf("CONFIG_CNTL: %08lx\n", config_cntl);

    // exit application and return to DOS
    exit (0);
}

