/*==========================================================================
* DRAWRECT.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved*
*                                                                          *
* PGL functions to draw filled rectangles.                                 *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  pgl_solidrect - draw a solid filled rectangle

  Draw a solid filled rectangle at (x, y) of size (width, height) using the
  current mix and source. This function can be used to perform the blit &
  fill part of a polygon fill.
-------------------------------------------------------------------------- */
void pgl_solidrect(int x, int y, int width, int height)
{
    unsigned long temp, rotation;

    // save vital regs
    PGL_waitforidle();
    temp = regr(DST_CNTL);

    // draw filled rectangle
    if (PGL_modecfg.bpp == 24)
    {
        // set 24 bpp rotation value
        rotation = pgl_get24bpprotation(x);

        regw(DST_CNTL, (temp & 0x80) | rotation |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT |
                       DST_Y_TILE | DST_X_TILE);
        regw(DST_Y_X, ((unsigned long)(x * 3) << 16) | y);
        regw(DST_HEIGHT_WIDTH, ((unsigned long)(width * 3) << 16) | height);
    }
    else
    {
        regw(DST_CNTL, (temp & 0x80) |
                       DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT |
                       DST_Y_TILE | DST_X_TILE);
        regw(DST_Y_X, ((unsigned long)(x) << 16) | y);
        regw(DST_HEIGHT_WIDTH, ((unsigned long)(width) << 16) | height);
    }

    // restore
    regw(DST_CNTL, temp);
}

/* --------------------------------------------------------------------------
  PGL_clearscreen - clear the screen

  The region is cleared by drawing a solid filled BLACK rectangle at (x, y)
  of size (width, height). The foreground color and mix are temporarly
  overriden for the operation.
-------------------------------------------------------------------------- */
void PGL_clearscreen(int x, int y, int width, int height)
{
    int mix;

    mix = PGL_getfgmix();
    PGL_setfgmix(ZERO_MIX);
    pgl_solidrect(x, y, width, height);
    PGL_setfgmix(mix);
}

/* --------------------------------------------------------------------------
  PGL_drawrect - draw a rectangle with the current fill pattern

  Draw a filled rectangle at (x, y) of size (width, height) using the current
  color and mix.
-------------------------------------------------------------------------- */
void PGL_drawrect(int x, int y, int width, int height)
{
    unsigned long temp1, temp2, temp3, temp4, temp5, temp6;
    unsigned long offset;
    int i, j;

    // Wait for engine idle to read vital registers for saving
    PGL_waitforidle();

    /* perform SOLID-FILL */
    if (PGL_attr.fillsolid == 1) /* do a solid fill */
    {
        // save vital regs
        temp1 = regr(DP_SRC);

        // set color source for rectangle fill
        regw(DP_SRC, FRGD_SRC_FRGD_CLR);

        // draw filled rectangle
        pgl_solidrect(x, y, width, height);

        // restore
        regw(DP_SRC, temp1);
    }
    else                         /* do a patterned fill */
    {
        // calculate qword offset to start of off-screen memory
        offset = pgl_getxyoffset(0, PGL_attr.srcrect_y) / 8;

        // save vital regs
        temp1 = regr(DST_OFF_PITCH);
        temp2 = regr(SRC_OFF_PITCH);
        temp3 = regr(DP_PIX_WIDTH);
        temp4 = regr(DP_SRC);
        temp5 = regr(SRC_CNTL);
        temp6 = regr(DP_MIX);

        // place patterned source in off-screen memory
        regw(DST_OFF_PITCH, (temp1 & 0xffc00000) | offset);
        if (PGL_modecfg.bpp == 4)
        {
            regw(DP_PIX_WIDTH, 0);
        }
        else
        {
            regw(DP_PIX_WIDTH, BYTE_ORDER_LSB_TO_MSB);
        }
        regw(DP_SRC, MONO_SRC_HOST | FRGD_SRC_FRGD_CLR);
        regw(DP_MIX, FRGD_MIX_ONE | BKGD_MIX_ZERO);
        regw(DST_Y_X, 0);
        regw(DST_HEIGHT, (unsigned long)PGL_attr.fillpatternheight);
        if (PGL_modecfg.bpp == 24)
        {
            regw(DST_WIDTH, 96);
            for (i = 0; i < PGL_attr.fillpatternheight; i++)
            {
                for (j = 0; j < 3; j++)
                {
                    PGL_waitforfifo(1);
                    regw(HOST_DATA0, PGL_attr.fillpattern[(i*3) + j]);
                }
            }
        }
        else
        {
            regw(DST_WIDTH, 32);
            for (i = 0; i < PGL_attr.fillpatternheight; i++)
            {
                PGL_waitforfifo(1);
                regw(HOST_DATA0, PGL_attr.fillpattern[i]);
            }
        }

        // setup engine to use source in blit mode
        PGL_waitforidle();
        regw(DST_OFF_PITCH, temp1);
        regw(SRC_OFF_PITCH, (temp2 & 0xffc00000) | offset);
        if (PGL_modecfg.bpp == 4)
        {
            regw(DP_PIX_WIDTH, temp3 & 0x000000ff);
        }
        else
        {
            regw(DP_PIX_WIDTH, (temp3 & 0x000000ff) | BYTE_ORDER_LSB_TO_MSB);
        }
        regw(SRC_Y_X, 0);
        regw(SRC_CNTL, SRC_LINE_X_LEFT_TO_RIGHT | SRC_PATTERN_ENABLE);
        regw(SRC_HEIGHT1, (unsigned long)PGL_attr.fillpatternheight);
        if (PGL_modecfg.bpp == 24)
        {
            regw(SRC_WIDTH1, (unsigned long)(PGL_attr.fillpatternwidth * 3));
        }
        else
        {
            regw(SRC_WIDTH1, (unsigned long)PGL_attr.fillpatternwidth);
        }
        regw(DP_SRC, MONO_SRC_BLIT | FRGD_SRC_FRGD_CLR);
        regw(DP_MIX, temp6);

        // draw pattern filled rectangle
        pgl_solidrect(x, y, width, height);

        // restore
        PGL_waitforfifo(5);
        regw(DST_OFF_PITCH, temp1);
        regw(SRC_OFF_PITCH, temp2);
        regw(DP_PIX_WIDTH, temp3);
        regw(DP_SRC, temp4);
        regw(SRC_CNTL, temp5);
    }
}

